import {ref} from 'vue';
import {useApi} from 'composable/api';
import {useDatetime} from 'composable/datetime';

/**
 * Contiene funciones y propiedades de un objeto de datatable.
 */
export function useDatatable(config) {
	
	// Número de registros visibles por defecto.
	let rows = 5;
	
	// Se define un objeto datatable que contiene las configuraciones
	let datatable = ref({
		// API
		api: useApi(),
		// Datetime format.
		datetime: useDatetime(),
		// URL de la que mandará a traer los datos.
		url: config['url'],
		// Expresiones where adicionales.
		additionalWhere: config['additionalWhere'] ? config['additionalWhere'] : '',
		// Privilegios sobre los registros del datatable.
		privileges: config['privileges'] ? config['privileges'] : [],
		// Llave identificadora de cada uno de los datos del datatable.
		dataKey: config['dataKey'] ? config['dataKey'] : null,
		// Columnas que se mandan a traer de la API.
		data: config['data'] ? config['data'] : [],
		// Arreglo con los valores dataKey de los registros que hayan sido seleccionados.
		selection: [],
		// Controla si se muestra la columna de los índices de cada registro.
		showIndex: config['showIndex'] != undefined ? config['showIndex'] : true,
		// Controla si se muestra la columna de selección con checkboxes o radiobuttons.
		showSelection: config['showSelection'] != undefined ? config['showSelection'] : false,
		// Controla si se muestra la columna de los botones de acción.
		showActions: config['showActions'] != undefined ? config['showActions'] : true,
		// Controla si se muestra el cuadro de busqueda.
		showSearch: config['showSearch'] != undefined ? config['showSearch'] : true,
		// Texto de la cabecera de columna de índices.
		headerIndex: '#',
		// texto de la cabecera de la columna de selección.
		headerSelection: '',
		// Texto de cabecera de columna de acciones.
		headerActions: 'Acciones',
		// Clases para aplicarle estilos a las tablas.
		tableClass: '',
		// Mecanismo de responsividad de las tablas al cambiar el tamaño de la pantalla.
		responsiveLayout: 'scroll',
		// Número de registros que se muestran por página.
		rows: rows,
		// Texto en el campo de búsqueda del datatable.
		searchPlaceholder: "Buscar...",
		// Texto que aparece cuando no hay resultados que mostrar.
		emptyMessage : "Ningún dato disponible en esta tabla",
		// Texto que se muestra mientras se está realizando una operación de ordenamiento, paginado, o búsqueda.
		loadingMessage : "",
		// Plantilla del paginador.
		paginatorTemplate: config['paginatorTemplate'] ? config['paginatorTemplate'] : "CurrentPageReport FirstPageLink PrevPageLink PageLinks NextPageLink LastPageLink RowsPerPageDropdown",
		// Texto de paginado.
		currentPageReportTemplate: "Mostrando {first} a {last} de {totalRecords} registros",
		// Combo con el número de registros posibles a mostrar por página.
		rowsPerPageOptions: [5,10,20,30,50],
		// Bandera que indica si se está cargando información.
		loading: false,
		// Arreglos con los datos de la tabla.
		value: [],
		// Total de registros de toda la tabla.
		totalRecords: 0,
		// Filtros de búsqueda.
		filters: {
			global: {
				value: ''
			}
		},
		// Parámetros de estado actual del datatable.
		lazyParams: {
			first: 0,
			rows: rows,
			sortField: null,
			sortOrder: null,
			filters: {global: {value: ''}}
		},
		// Retardo en milisegundos para la ejecución de una búsqueda.
		delaySearchExecution: 1500,
		// Objeto que contiene la llamada a la búsqueda.
		search: null,
		// Obtiene los datos para rellenar el datatable.
		getValue(event) {
			// Tiempo en que se enviará la petición al servidor.
        	let time = 0;
            // Si es un evento de ordenamiento o paginamiento.
            if(event != undefined) {
            	if(event['originalEvent'] != undefined) { this.lazyParams = event; }
            	// Si se está presionando enter. 
                else if(event['key'] == 'Enter') { this.lazyParams.filters = JSON.parse(JSON.stringify(this.filters)); }
            	// Si es evento de filtrado, y ha cambiado el texto a buscar.
                else if(this.lazyParams.filters.global.value.trim() != this.filters.global.value.trim()) { 
                    this.lazyParams.filters = JSON.parse(JSON.stringify(this.filters)); 
                    time = this.delaySearchExecution;
                }
                // Si se presionaron teclas de control o espacio, no programa una nueva búsqueda.
                else { return; }
            }
            // Se limpia la petición que hayan quedado en cola.
            if(this.search) { 
                clearTimeout(this.search); 
                this.search = null;
            }
            // Se cargan los datos.
            this.search = setTimeout(() => {
            	this.loading = true;
            	this.value = [];
            	// Se obtienen las expresiones de selección.
            	let select = this.data.map(obj => obj.name).join('|');
            	// Se obtienen las expresiones de ordenamiento.
            	let order = '';
            	if(this.lazyParams.sortField) {
            		let cols = [];
            		const dir = this.lazyParams.sortOrder == -1 ? 'desc' : 'asc';
            		for(const c of this.lazyParams.sortField.split('|')) {
            			cols.push(`${c} ${dir}`);
            		}
            		order = cols.join('|');
            	}
            	// Se obtienen las expresiones de filtrado.
            	let v = this.lazyParams.filters.global.value.trim();
            	let where = '';
            	if(v) {
            		where = this.data.filter(obj => obj.visible !== false).map(obj => {
            			let cols = [];
            			for(const c of obj.name.split('|')) {
                			cols.push(`${c} li %${v}%`);
                		}
            			return cols.join('|or|');
            		}).join('|or|')
            		where = '(|'+where+'|)';
            	}
            	where = where ? (this.additionalWhere ? where+'|and|'+this.additionalWhere : where) : this.additionalWhere;
            	
            	// Se hace la llamada para obtener los datos.
    			this.api.request('GET', this.url, null, {
    				'select': select,
                	'where': where,
                	'order': order,
                	'offset': this.lazyParams.first,
                	'limit': this.lazyParams.rows
    			}).then(response => {
    				this.totalRecords = parseInt(response.totalRecords);
                    this.value = response.data;
                	this.loading = false;
                });
            }, time);
		},
		// Devuelve un botón de creación si está en los privilegios.
		getTopButtons: config['getTopButtons'] ? config['getTopButtons'] : function() {
			let topButtons = [];
			if(this.privileges) {
				for(const [k, v] of Object.entries(this.privileges)) {
					if(v.operation == 1) {
						topButtons.push({
							button: false,
							href: v.url,
							buttonClass: v.button,
							label: v.label,
							text: '',
							iconClass: v.icon,
							disabled: false,
							dataAttributes: this.getDataAttributes({
								'operation': v.operation
							})
						});
						break;
					}
				}
			}
			return topButtons;
		},
		// Devuelve los botones de acción sobre cada registro.
		getActionButtons: config['getActionButtons'] ? config['getActionButtons'] : function(row) {
			let actionButtons = [];
			if(this.privileges) {
				for(const [k, v] of Object.entries(this.privileges)) {
					if(v.operation != 1) {
						actionButtons.push({
							button: false,
							href: v.url+'/'+row[this.dataKey],
							target: '_self',
							buttonClass: v.button,
							label: v.label,
							iconClass: v.icon,
							disabled: false,
							dataAttributes: this.getDataAttributes({
								'operation': v.operation
							})
						});
					}
				}
			}
			return actionButtons;
		},
		// Devuelve un objeto con múltiple atributos data.
		getDataAttributes: function(attributes) {
			let dataAttributes = {};
			if(attributes) {
				for(const [k, v] of Object.entries(attributes)) {
					dataAttributes['data-'+k] = v;
				}
			}
			return dataAttributes;
		}
	});
	
	// Se devuelve el objeto.
	return datatable;
	
}