import {useDatatable} from 'composable/datatable';
import {useDatetime} from 'composable/datetime';

export default {
	components: {
		'p-datatable': DataTable,
		'p-column': Column
	},
	props: [
		'config'											// Parámetro de configuraciones que puede recibir el datatable.
	],
	data() {
		return {
			datatable: {}									// Contiene el composable useDatatable.
		}
	},
	created() {
		// Se asignan las configuraciones.
		let conf = {
			immediate: this.config.immediate,					// Indica si al momento de crearse el datatable carga los datos.
			base: this.config.base,								// URL base para llamada API.
        	url: this.config.url,								// URL de la que tomará los datos.
        	additionalWhere: this.config.additionalWhere,		// Expresiones where adicionales.
        	privileges: this.config.privileges,					// Conjunto de privilegios con los cuales se dibujarán los botones.
            showIndex: this.config.showIndex,					// Indica si se muestra la columna de índices.
            showSelection: this.config.showSelection,			// Indica si se muestra la columna de selección.
            showActions: this.config.showActions,				// Indica si se muestra la columna de botones de acción.
        	dataKey: this.config.dataKey,						// Indica cual es el dato ID de los registros de la tabla.
        	data: this.config.data,								// Indica la configuración de columnas a mostrar.
        	paginatorTemplate: this.config.paginatorTemplate	// Se establece los controles de paginación que se mostrarán
		};
		// Se asigna la función de generación de botones.
		if(this.config.getActionButtons) {
			conf['getActionButtons'] = this.config.getActionButtons;
		}
		// Sobreescribe el método por default de botón de crear.
		if(this.config.getTopButtons) {
			conf['getTopButtons'] = this.config.getTopButtons;
		}
    	// Se carga el objeto datatable.
    	this.datatable = useDatatable(conf);
    	// Carga los datos por primera vez.
    	if(this.config['immediate'] !== false) {
    		this.datatable.getValue();
    	}
    },
    methods: {
    	// Dispara la actualización de los datos del datatable.
    	getValue() {
    		this.datatable.getValue();
    	}
    },
	template: `
		<p-datatable 
				class="p-datatable-sm" :responsive-layout="datatable.responsiveLayout"
				:value="datatable.value" :total-records="datatable.totalRecords" v-model:filters="datatable.filters"
				:paginator="true" :paginator-template="datatable.paginatorTemplate" :current-page-report-template="datatable.currentPageReportTemplate" :rows-per-page-options="datatable.rowsPerPageOptions" :rows="datatable.rows" :loading="datatable.loading"
				:lazy="true" @page="datatable.getValue($event)" @sort="datatable.getValue($event)"> 
				<template #empty><div class="text-center">{{datatable.emptyMessage}}</div></template>
				<template #loading><div class="text-center">{{datatable.loadingMessage}}</div></template>
				<template #header>
					<div class="row">
						<div class="col-3 col-md-9">
							<template v-for="(action, actionIndex) in datatable.getTopButtons()">
								<!-- Por defecto todos los botones son enlaces -->
								<a v-if="action.button !== true" :href="action.href" :target="action.target" :class="[action.buttonClass, {'disabled': action.disabled}]" 
									:title="action.label" style="margin: 2px" v-bind="datatable.getDataAttributes(action.dataAttributes)">
									<i v-if="action.iconClass" :class="action.iconClass" style="pointer-events: none"></i>
									{{action.text ? '&nbsp;'+action.text : ''}}
								</a>
								<!-- Si se especifica que sea tipo boton, se manda la llave button en true -->
								<button v-if="action.button === true" type="button" :class="action.buttonClass" 
									:title="action.label" :disabled="action.disabled" style="margin: 2px" @click="this.$emit(action.event ? action.event : 'action', $event)"
									v-bind="datatable.getDataAttributes(action.dataAttributes)">
									<i v-if="action.iconClass" :class="action.iconClass" style="pointer-events: none"></i>
									{{action.text ? '&nbsp;'+action.text : ''}}
								</button>
							</template>
						</div>
						<div class="col-9 col-md-3">
							<input type="text" class="form-control form-control-sm" :placeholder="datatable.searchPlaceholder" v-model="datatable.filters['global'].value" @keyup="datatable.getValue($event)" :readonly="datatable.loading">
						</div>
					</div>
				</template> 
				<p-column :header="datatable.headerIndex" style="width:1%" v-if="datatable.showIndex">
					<template #body="slotProps">{{datatable.lazyParams.first + (slotProps.index + 1)}}</template>
				</p-column>
				<p-column :header="datatable.headerSelection" style="width:1%" v-if="datatable.showSelection">
					<template #body="slotProps">
						<input type="checkbox" :value="slotProps.data[this.datatable.dataKey]" v-model="this.datatable.selection" 
							@click="this.$emit('selection', $event, slotProps.data)" />
					</template>
				</p-column>
				<template v-for="(column, columnIndex) in datatable.data">
					<p-column v-if="column['visible'] === false ? false : true" :field="column.name" :header="column['header'] ? column['header'] : ''" :sortable="column['sortable'] === false ? false : true">
						<template #body="slotProps" v-if="column['getTemplate']">
							<div v-html="column.getTemplate(slotProps.data)"></div>
						</template>
					</p-column>
				</template>
				<p-column :header="datatable.headerActions" style="width:1%; white-space: nowrap; text-align: right" v-if="datatable.showActions === true">
					<template #body="slotProps">
						<template v-for="(action, actionIndex) in datatable.getActionButtons(slotProps.data)">
							<!-- Por defecto todos los botones son enlaces -->
							<a v-if="action.button !== true" :href="action.href" :target="action.target" :class="[action.buttonClass, {'disabled': action.disabled}]" 
								:title="action.label" style="margin: 2px" v-bind="datatable.getDataAttributes(action.dataAttributes)">
								<i :class="action.iconClass" style="pointer-events: none"></i>
							</a>
							<!-- Si se especifica que sea tipo boton, se manda la llave button en true -->
							<button v-if="action.button === true" type="button" :class="action.buttonClass" 
								:title="action.label" :disabled="action.disabled" style="margin: 2px" @click="this.$emit('action', $event, slotProps.data)"
								v-bind="datatable.getDataAttributes(action.dataAttributes)">
								<i :class="action.iconClass" style="pointer-events: none"></i>
							</button>
						</template>
					</template>
				</p-column>
			</p-datatable>
	`
}