<?php

namespace Application\Library\Listener;

use Laminas\EventManager\AbstractListenerAggregate;
use Laminas\View\Resolver\TemplateMapResolver;
use Laminas\Filter\FilterInterface;
use Laminas\Filter\FilterChain;
use Laminas\Filter\Word\CamelCaseToDash;
use Laminas\Filter\StringToLower;
use Laminas\Mvc\MvcEvent;
use Laminas\EventManager\EventManagerInterface;
use Application\Library\Authentication\Session;
use Application\Library\Project\Server;
use Application\Library\Api\Api;
use Laminas\Http\Request;
use Application\Library\Authentication\Menu;
use Application\Library\Complements\MyLogger;
use Exception;

/**
 * It manages all actions in a MVC cycle.
 * @author workstation2
 *
 */
class MvcListener extends AbstractListenerAggregate
{
    /**
     * @var TemplateMapResolver
     */
    private $templateMapResolver;

    /**
     * @var FilterInterface
     */
    private $filter;

    /**
     * Constructor.
     * @param TemplateMapResolver $templateMapResolver
     */
    public function __construct(TemplateMapResolver $templateMapResolver)
    {
        $this->templateMapResolver = $templateMapResolver;
        $this->filter = (new FilterChain())
            ->attach(new CamelCaseToDash())
            ->attach(new StringToLower());
    }

    /**
     * Attach the listeners.
     * {@inheritDoc}
     * @see \Laminas\EventManager\ListenerAggregateInterface::attach()
     */
    public function attach(EventManagerInterface $events, $priority = 1)
    {
        $this->listeners[] = $events->attach(MvcEvent::EVENT_DISPATCH_ERROR, [$this, 'error']);
        $this->listeners[] = $events->attach(MvcEvent::EVENT_DISPATCH, [$this, 'layout']);
        $this->listeners[] = $events->attach(MvcEvent::EVENT_DISPATCH, [$this, 'authorize']);
        $this->listeners[] = $events->attach(MvcEvent::EVENT_DISPATCH, [$this, 'default']);
        $this->listeners[] = $events->attach(MvcEvent::EVENT_DISPATCH, [$this, 'breadcrumb']);
    }

    /**
     * Handles any type of error during dispatching.
     * @param MvcEvent $event
     */
    public function error(MvcEvent $event)
    {
        // Get login URL.
        $generalUrl = $event->getRouter()->assemble([], ['name' => 'publicGeneral']);
//         var_dump($event);
//         exit();
        // In case of error, redirects to login.
        Server::redirectTo($generalUrl);
    }

    /**
     * Checks authorization to system's views.
     * @param MvcEvent $event
     */
    public function authorize(MvcEvent $event)
    {
        if (!isset($_COOKIE['idIdioma'])) {
            setcookie('idIdioma', '1', 0, '/');
            setcookie('tokenSedetur', '', time() + 86400, '/', '', true, true);
            $_COOKIE['idIdioma'] = 1;
        }
        // Get home, login and logout URL.
        $router = $event->getRouter();
        $homeURL = $router->assemble([], ['name' => 'home']);
        $loginURL = $router->assemble([], ['name' => 'login']);
        $logoutURL = $router->assemble([], ['name' => 'logout']);
        $generalUrl = $router->assemble([], ['name' => 'publicGeneral']);
        $templateBaseUrl = $router->assemble([], ['name' => 'template']);

        // Gets the matched route.
        $routeMatch = $event->getRouteMatch();
        if (!$routeMatch) {
            Server::redirectTo($loginURL);
        }

        // If user is loggin out, then clean session and redirect to login.
        if ($routeMatch->getMatchedRouteName() == 'logout') {
            Session::clean();
            Server::redirectTo($loginURL);
        }

        // If user is loggin out, then clean session and redirect to login.
        // if($routeMatch->getMatchedRouteName() == 'publicGeneral') {
        //     $event->getViewModel()->setTemplate('general/layout');

        // }

        // Validates wich affiliate is using the system.
        if ($routeMatch->getMatchedRouteName() == 'login') {
            return;
//             $affiliate = $routeMatch->getParam('affiliate');
//             if($affiliate) {
//                 if($affiliate == 'favicon.ico') {
//                     return;
//                 }
//                 if($affiliate != @$_COOKIE['affiliate']) {
//                     setcookie('affiliate', trim($affiliate), time()+(60*60*24*30), '/');
//                     Server::redirectTo($logoutURL);
//                 }
//                 $authResult = Api::request('/authentication/'.$affiliate, Request::METHOD_GET, null, false);
//                 if($authResult->getStatusCode() != 200) {
//                     $event->getViewModel()->setTemplate('layout/affiliate');
//                     return;
//                 }
//             } else if(@$_COOKIE['affiliate']) {
//                 Server::redirectTo($loginURL);
//             } else {
//                 $event->getViewModel()->setTemplate('layout/affiliate');
//                 return;
//             }
        }

        // Obtain module name.
        $controller = $routeMatch->getParam('controller');
        if (!$controller) {
            Server::redirectTo($homeURL);
        }
        $module = substr($controller, 0, strpos($controller, '\\'));
        if ($module == 'General') {
            $loginData = [
                'frmCorreo' => 'Public',
                'frmContraseña' => 'Superadmin_2023#$%',
                'frmUidRedSocial' => null
            ];
            $loginResult = Api::request('/login', request::METHOD_POST, $loginData, false);
            $login = json_decode($loginResult->getBody(), true);
            setcookie('accessToken', $login['accessToken'], 0, '/');
            setcookie('refreshToken', $login['refreshToken'], 0, '/');
            setcookie('sessionToken', $login['sessionToken'], 0, '/');

            $_COOKIE['accessToken'] = $login['accessToken'];
            $_COOKIE['refreshToken'] = $login['refreshToken'];
            $_COOKIE['sessionToken'] = $login['sessionToken'];

            $event->getViewModel()->setTemplate('general/layout');
            // Fetched user's privileges
            $footerResult = Api::request('/footer', Request::METHOD_GET, [
                'where' => 'frmIdIdioma eq ' . $_COOKIE['idIdioma']
            ]);
            $footerJson = json_decode($footerResult->getBody(), true)['data'];
            $_SESSION['footer'] = $footerJson[0]['frmContenidoFooter'];
            $bannerResult = Api::request('/banner', Request::METHOD_GET, [
                'where' => 'frmIdIdioma eq ' . $_COOKIE['idIdioma']
            ]);
            $bannerJson = json_decode($bannerResult->getBody(), true)['data'];
            $_SESSION['banner'] = $bannerJson[0]['frmContenidoBanner']['banner'][0];

            $accesosResult = Api::request('/accesos-directos', Request::METHOD_GET, [
                'where' => 'frmIdIdioma eq ' . $_COOKIE['idIdioma']
            ]);
            $accesosJson = json_decode($accesosResult->getBody(), true)['data'];
            $_SESSION['accesosDirectos'] = $accesosJson;

            $menuPublicoResult = Api::request('/menu-publico', Request::METHOD_GET, [
                'where' => 'frmIdIdioma eq ' . $_COOKIE['idIdioma']
            ]);
            $menuJson = json_decode($menuPublicoResult->getBody(), true)['data'];
            $menu = [];
            foreach ($menuJson as $registro) {
                $menu[$registro['frmIdMenuPublico']] = [
                    'text' => $registro['frmEtiquetaMenuPublico'],
                    'class' => $registro['frmIconoMenuPublico'],
                    'link' => $registro['frmUrlMenuPublico'],
                    'show_condition' => $registro['frmVisibleMenuPublico'],
                    'parent' => $registro['frmPadreMenuPublico']
                ];
            }
            $_SESSION['menuPublico'] = $menu;
            $etiquetasResult = Api::request('/etiquetas', Request::METHOD_GET);
            $etiquetasJson = json_decode($etiquetasResult->getBody(), true)['data'];
            $_SESSION['etiquetas'] = $etiquetasJson;
            // If user comes from a succesful login attempt, user data and privileges are fetched and set into $_SESSION.
            // if (Session::logedIn()) {

            //     // Fetched user data from the access token.
            //     $authResult = Api::request('/login', Request::METHOD_GET);
            //     $payload = json_decode($authResult->getBody(), true);
            //     if (empty($payload['id'])) {
            //         Server::redirectTo($logoutURL);
            //     }
            //     $_SESSION['user'] = [
            //         'id' => $payload['id'],
            //         'user' => $payload['user'],
            //         'citizen' => $payload['name'] . $payload['lastName'] . $payload['maternalName'],
            //         'initials' => 'U'
            //     ];
            //     return;
            // }
            // else{
            return;
            // }
        }

        // If session does not exist.
        if (empty($_SESSION['user'])) {

            // If user tries to access a protected resource without session.
            if ($module != 'Authentication') {

                // If user comes from a succesful login attempt, user data and privileges are fetched and set into $_SESSION.
                if (Session::logedIn()) {

                    // Fetched user data from the access token.
                    $authResult = Api::request('/login', Request::METHOD_GET);
                    $payload = json_decode($authResult->getBody(), true);
                    if (empty($payload['id'])) {
                        Server::redirectTo($logoutURL);
                    }
                    $_SESSION['user'] = [
                        'id' => $payload['id'],
                        'user' => $payload['user'],
                        'citizen' => $payload['name'] . ' ' . $payload['lastName'] . ' ' . $payload['maternalName'],
                        'initials' => 'U'
                    ];

                    // Fetched user's privileges
                    $privilegesResult = Api::request('/users-privileges', Request::METHOD_GET, [
                        'where' => 'frmUserIdUserPrivilege eq ' . $payload['id'] . '|and|frmVisibleMenu eq 1'
                    ]);
                    $privileges = json_decode($privilegesResult->getBody(), true)['data'];
                    $menu = new Menu($privileges);
                    $menu->generate();

                } // If user does not come from login form, cookies and session are cleaned and redirected to login.
                else {
                    Session::clean();
                    Server::redirectTo($loginURL);
                }
            }

        } // If session exists.
        else {

            // Validates that all three tokens exist.
            if (empty($_COOKIE['accessToken']) || empty($_COOKIE['refreshToken']) || empty($_COOKIE['sessionToken'])) {

                // Unset the whole session variable.
                Session::clean();

                // User is redirected to login.
                Server::redirectTo($loginURL);
            }

            // If user is trying to access to Authentication module views, then is redirected to home.
            if ($module == 'Authentication') {
                Server::redirectTo($homeURL);
            }

            // validates if user has acces to the current URL.
            $noAuthRequired = [
                $homeURL,
                $templateBaseUrl
            ];
            $url = '/' . implode('/', array_slice(explode('/', trim(explode('?', $_SERVER['REQUEST_URI'])[0], '/')), 0, 3));
            if (!in_array($url, $noAuthRequired)) {
                if (!Menu::userHasAccess($url)) {
                    Server::redirectTo($homeURL);
                }
            }
        }

    }

    /**
     * Set the layout according to the module or controller.
     * @param MvcEvent $event
     */
    public function layout(MvcEvent $event)
    {

        // Get and check the route match object.
        $routeMatch = $event->getRouteMatch();
        if (!$routeMatch) {
            return;
        }

        // Get and check the parameter for current controller.
        $controller = $routeMatch->getParam('controller');
        if (!$controller) {
            return;
        }

        // Extract module name.
        $module = substr($controller, 0, strpos($controller, '\\'));

        // Convert the module name from camel case to a lower string with dashes.
        $name = 'layout/' . $this->filter->filter($module);

        // Has the resolver an entry layout/ with the given name?
        if (!$this->templateMapResolver->has($name)) {
            return;
        }

        // Get root view model.
        $layoutViewModel = $event->getViewModel();

        // Rendering without layout?
        if ($layoutViewModel->terminate()) {
            return;
        }

        // Change template.
        $layoutViewModel->setTemplate($name);

    }

    /**
     * Sets default values for some variables in the view and layout.
     * @param MvcEvent $event
     */
    public function default(MvcEvent $event)
    {
        // Data retrieved.
        $actionViewModel = $event->getViewModel()->getChildren()[0];
        $variables = $actionViewModel->getVariables();
        $action = $event->getRouteMatch()->getParam('action');
        $defaultId = $event->getRouteMatch()->getParam('id') ?: 0;
        $target = $event->getTarget();

        // Default view name according to the action called.
        switch ($action) {
            case 'index':
                $defaultView = 'Inicio';
                break;
            case 'create':
                $defaultView = 'Crear';
                break;
            case 'edit':
                $defaultView = 'Editar';
                break;
            case 'details':
                $defaultView = 'Detalles';
                break;
            case 'delete':
                $defaultView = 'Eliminar';
                break;
            default:
                $defaultView = $action;
        }

        // Default method according to the action called.
        switch ($action) {
            case 'create':
                $defaultMethod = Request::METHOD_POST;
                break;
            case 'edit':
                $defaultMethod = Request::METHOD_PUT;
                break;
            case 'details':
                $defaultMethod = Request::METHOD_GET;
                break;
            case 'delete':
                $defaultMethod = Request::METHOD_DELETE;
                break;
            default:
                $defaultMethod = Request::METHOD_GET;
        }

        // If an index view is called, then privileges for a datatable are queried.
        $defaultPrivileges = [];
        if ($action == 'index') {
            $menu = new Menu();
            $children = $menu->getChildrenByUrl($_SERVER['REQUEST_URI']) ?: [];
            foreach ($children as $c) {
                $defaultPrivileges[] = [
                    'operation' => $c['frmOperationMenu'],
                    'url' => $c['frmUrlMenu'],
                    'icon' => $c['frmIconMenu'],
                    'button' => $c['frmButtonMenu'],
                    'label' => $c['frmLabelMenu']
                ];
            }
        }

        // Sets default values to use in layout and view.
        $actionViewModel->setVariables([
            'controller' => $target->controller,
            'view' => @$variables['view'] ?: $defaultView,
            'method' => @$variables['method'] ?: $defaultMethod,
            'privileges' => json_encode($defaultPrivileges),
            'id' => $defaultId
        ]);

    }

    /**
     * Loads the data for generating the breadcrumb.
     * @param MvcEvent $event
     */
    public function breadcrumb(MvcEvent $event)
    {
        // Gets the controller instance
        $target = $event->getTarget();
        // Assigns the variables.
        $event->getViewModel()->setVariables([
            'module' => $target->module,
            'controller' => $target->controller,
            'view' => $event->getViewModel()->getChildren()[0]->getVariables()['view']
        ]);
    }

}