<?php
namespace Application\Model\Security\Users;

use Application\Library\Complements\MyLogger;
use Application\Library\Model\Table;
use Laminas\Db\TableGateway\TableGateway;
use Laminas\ServiceManager\ServiceManager;
use Laminas\Db\Sql\Select;
use Application\Library\Security\Password;
use Laminas\Db\Sql\Insert;
use Laminas\Db\Sql\Update;
use Laminas\Db\Sql\Delete;
use Application\Model\Catalog\Employees\EmployeesTable;
use Application\Model\Catalog\Genders\GendersTable;
use Application\Model\Security\UsersPrivileges\UsersPrivilegesTable;
use Application\Model\Security\UsersSections\UsersSectionsTable;
use Application\Model\Security\UsersSocialNetworks\UsersSocialNetworksTable;
use Laminas\Db\Sql\Join;

class UsersTable extends Table
{
    /**
     * Constructor
     * @param TableGateway $tableGateway
     * @param ServiceManager $sm
     */
    public function __construct($tableGateway, $sm)
    {
        parent::__construct($tableGateway, $sm);
    }
    
    /**
     * Fetch a collection.
     * @param array $params
     * @param \Countable $resultSetPrototype
     * @return \Countable
     */
    public function fetchAll($params = [])
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdUsuario' => 'id',
            'frmIdNombreUsuarioUsuario' => 'nombre_usuario',
            'frmActivoUsuario' => 'activo'
        ]);
        
        $employeesTable = $this->sm->get(EmployeesTable::class);
        $select->join($employeesTable->tableGateway->table,
            $this->on($this->column('id_empleado'), $employeesTable->column('id')), [
                'frmNombreEmpleado' => 'nombre',
                'frmApellidosEmpleado' => 'apellidos'
        ]);
        
//         $usersSectionsTable = $this->sm->get(UsersSectionsTable::class);
//         $select->join($usersSectionsTable->tableGateway->table,
//             $this->on($this->column('id'), $usersSectionsTable->column('user_id')), [
//                 'frmStateIdUserSection' => 'state_id'
//             ], Join::JOIN_LEFT);
        
        $select->order($this->asc('nombre_usuario'));
        
        return $this->selectCollection($this->tableGateway, $select, $params);
    }
    
    /**
     * Fetch a resource.
     * @param int $id
     * @return \Application\Library\Model\Result
     */
    public function fetch($id)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdUsuario' => 'id',
            'frmEmployeeIdUser' => 'id_empleado',
            'frmIdPerfilUsuario' => 'profile_id',
            'frmIdNombreUsuarioUsuario' => 'nombre_usuario'
        ]);
        
        $employeesTable = $this->sm->get(EmployeesTable::class);
        $select->join($employeesTable->tableGateway->table,
            $this->on($this->column('id_empleado'), $employeesTable->column('id')), [
                'frmNombreEmpleado' => 'nombre',
                'frmApellidosEmpleado' => 'apellidos'
            ]);
        
//         $usersSectionsTable = $this->sm->get(UsersSectionsTable::class);
//         $select->join($usersSectionsTable->tableGateway->table,
//             $this->on($this->column('id'), $usersSectionsTable->column('user_id')), [
//                 'frmIdUserSection' => 'id'
//             ]);
        
        $select->where->equalTo($this->column('id'), $id);
        
        return $this->selectEntity($this->tableGateway, $select);
    }
    
    /**
     * Creates a resource.
     * @param array $data
     * @throws \Exception
     * @return mixed
     */
    public function create($data)
    {
        try {
            $this->connection->beginTransaction();
            
            $employee = $this->sm->get(EmployeesTable::class)->selectById($data['frmEmployeeIdUser']);
            
            $insert = new Insert($this->tableGateway->table);
            $insert->values([
                'id_empleado' => $data['frmEmployeeIdUser'],
                'profile_id' => $data['frmIdPerfilUsuario'],
                'nombre_usuario' => $employee->frmCorreoElectronicoEmpleado,
                'password' => Password::create($data['frmPasswordUsuario']),
                'activo' => 't',
                'user_id_creation' => $_SESSION['user']['id'],
                'creation_date' => date('Y-m-d'),
                'creation_time' => date('H:i:s')
            ]);
            $id = $this->insertWith($insert);
            
            $usersPrivilegesTable = $this->sm->get(UsersPrivilegesTable::class);
            foreach($data['frmPrivilegesUser'] as $privilege) {
                $usersPrivilegesTable->insert([
                    'frmUserIdUserPrivilege' => $id,
                    'frmMenuIdUserPrivilege' => $privilege
                ]);
            }
            
            $usersSectionsTable = $this->sm->get(UsersSectionsTable::class);
            $usersSectionsTable->create([
                'frmUserIdUserSection' => $id
            ]);
            
            $this->connection->commit();
            return $id;
        } catch (\Exception $e) {
            $this->connection->rollback();
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * Creates a resource.
     * @param array $data
     * @throws \Exception
     * @return mixed
     */
    public function createFromWeb($data)
    {
        $insert = new Insert($this->tableGateway->table);
        $insert->values([
            'id_empleado' => $data['frmIdUsuarioCatalogo'],
            'id_perfil' => '1',
            'id_red_social' => $data['frmIdRedSocialUsuario'],
            'nombre_usuario' => $data['frmNombreUsuario'],
            'password' => $data['frmPasswordUsuario'] ? Password::create($data['frmPasswordUsuario']) : null,
            'uid' => $data['frmUidRedSocialUsuario'],
            'activo' => '1',
            'id_usuario_creacion' => '1',
            'fecha_creacion' => date('Y-m-d'),
            'hora_creacion' => date('H:i:s')
        ]);

        $id = $this->insertWith($insert);
        return $id;
    }
    
    /**
     * Updates a resource.
     * @param int $id
     * @param array $data
     * @throws \Exception
     * @return mixed
     */
    public function update($id, $data)
    {
        try {
            $this->connection->beginTransaction();
            
            $employee = $this->sm->get(EmployeesTable::class)->selectById($data['frmEmployeeIdUser']);
            
            $update = new Update($this->tableGateway->table);
            $set = [
                'id_empleado' => $data['frmEmployeeIdUser'],
                'profile_id' => $data['frmIdPerfilUsuario'],
                'user_id_modification' => $_SESSION['user']['id'],
                'modification_date' => date('Y-m-d'),
                'modification_time' => date('H:i:s')
            ];
            if($id != 1) {
                $set['nombre_usuario'] = $employee->frmCorreoElectronicoEmpleado;
            }
            $update->set($set);
            $update->where->equalTo($this->column('id'), $id);
            $this->tableGateway->updateWith($update);
            
            $usersPrivilegesTable = $this->sm->get(UsersPrivilegesTable::class);
            $usersPrivilegesTable->deleteByUserId($id);
            foreach($data['frmPrivilegesUser'] as $privilege) {
                $usersPrivilegesTable->insert([
                    'frmUserIdUserPrivilege' => $id,
                    'frmMenuIdUserPrivilege' => $privilege
                ]);
            }
            
            $this->connection->commit();
            return $id;
        } catch (\Exception $e) {
            $this->connection->rollback();
            throw new \Exception($e->getMessage());
        }
    }
    
    /**
     * Patch a resource.
     * @param int $id
     * @param array $data
     * @throws \Exception
     * @return mixed
     */
    public function patch($id, $data)
    {
        try {
            $this->connection->beginTransaction();
            
            $update = new Update($this->tableGateway->table);
            $update->set([
                'password' => Password::create($data['frmPasswordUsuario']),
                'user_id_modification' => $_SESSION['user']['id'],
                'modification_date' => date('Y-m-d'),
                'modification_time' => date('H:i:s')
            ]);
            $update->where->equalTo($this->column('id'), $id);
            $this->tableGateway->updateWith($update);
            
            $this->connection->commit();
            return $id;
        } catch (\Exception $e) {
            $this->connection->rollback();
            throw new \Exception($e->getMessage());
        }
    }
    
    /**
     * Deletes a resource. Attempts physical deletion, if not, a logic deletion is performed.
     * @param int $id
     */
    public function delete($id)
    {
        try {
            
            $this->connection->beginTransaction();
            
            $usersPrivilegesTable = $this->sm->get(UsersPrivilegesTable::class);
            $usersPrivilegesTable->deleteByUserId($id);
            
            $delete = new Delete($this->tableGateway->table);
            $delete->where->equalTo($this->column('id'), $id);
            $this->tableGateway->deleteWith($delete);
            
            $this->connection->commit();
            
        } catch (\Exception $e) {
            
            $this->connection->rollback();
            
            $update = new Update($this->tableGateway->table);
            $update->set(['activo' => 'f']);
            $update->where->equalTo($this->column('id'), $id);
            $this->tableGateway->updateWith($update);
            
        }
    }
    
    /**
     * Select resource by ID.
     * @param int $id
     * @return User
     */
    public function selectById($id)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdUsuario' => 'id',
            'frmIdNombreUsuarioUsuario' => 'nombre_usuario',
            'frmPasswordUsuario' => 'password',
            'frmActivoUsuario' => 'activo',
        ]);
        
        $select->where->equalTo($this->column('id'), $id);
        
        return $this->tableGateway->selectWith($select)->current();
    }
    
    /**
     * Select a resource by name.
     * @param string $name
     * @param int $omit
     * @return User
     */
    public function selectByName($name, $idOmit = null, $lowerCase = true)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdUsuario' => 'id',
            'frmIdNombreUsuarioUsuario' => 'nombre_usuario',
            'frmPasswordUsuario' => 'password',
            'frmUidUsuario' => 'uid',
            'frmActivoUsuario' => 'activo',
        ]);
        
        $employeesTable = $this->sm->get(EmployeesTable::class);
        $select->join($employeesTable->tableGateway->table,
            $this->on($this->column('id_empleado'), $employeesTable->column('id')), [
                'frmNombreEmpleado' => 'nombre',
                'frmApellidosEmpleado' => 'apellidos',
                'frmIdGeneroUsuarioEmpleado' => 'id_genero',
                'frmFechaNacimientoEmpleado' => 'fecha_nacimiento'
            ]);
        
        $select->where->equalTo($this->column('nombre_usuario'), $name);
        if($idOmit) {
            $select->where->notEqualTo($this->column('id'), $idOmit);
        }
        
        return $this->tableGateway->selectWith($select)->current();
    }
    
    /**
     * Select a resource by refresh token;
     * @param string $refreshToken
     * @return mixed
     */
    public function selectByRefreshToken($refreshToken)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdUsuario' => 'id',
            'frmIdNombreUsuarioUsuario' => 'nombre_usuario',
            'frmFechaHoraCreacionTokenUsuario' => 'token_creation_datetime'
        ]);
        
        $employeesTable = $this->sm->get(EmployeesTable::class);
        $select->join($employeesTable->tableGateway->table,
            $this->on($this->column('id_empleado'), $employeesTable->column('id')), [
                'frmNombreEmpleado' => 'nombre',
                'frmApellidosEmpleado' => 'apellidos'
            ]);
        
        $select->where->equalTo($this->column('refresh_token'), $refreshToken);
        
        return $this->tableGateway->selectWith($select)->current();
    }
    
    /**
     * Updates user's tokens.
     * @param int $id
     * @param string $accessToken
     * @param string $refreshToken
     */
    public function updateTokens($id, $accessToken, $refreshToken = null)
    {
        $update = new Update($this->tableGateway->table);
        $update->set([
            'access_token' => $accessToken,
            'refresh_token' => $refreshToken,
            'token_creation_datetime' => date('Y-m-d H:i:s')
        ]);
        $update->where->equalTo($this->column('id'), $id);
        $this->tableGateway->updateWith($update);
    }

    /**
     * Select a resource by name.
     * @param string $name
     * @param int $omit
     * @return User
     */
    public function selectByUid($params = [])
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdUsuario' => 'id',
            'frmIdNombreUsuarioUsuario' => 'nombre_usuario',
            'frmPasswordUsuario' => 'password',
            'frmUidUsuario' => 'uid',
            'frmIdRedSocialUsuario' => 'id_red_social',
            'frmActivoUsuario' => 'activo',
        ]);
        
        $employeesTable = $this->sm->get(EmployeesTable::class);
        $select->join($employeesTable->tableGateway->table,
            $this->on($this->column('id_empleado'), $employeesTable->column('id')), [
                'frmNombreEmpleado' => 'nombre',
                'frmApellidosEmpleado' => 'apellidos',
                'frmIdGeneroUsuarioEmpleado' => 'id_genero',
                'frmFechaNacimientoEmpleado' => 'fecha_nacimiento'
            ]);

        $usersSocialNetworksTable = $this->sm->get(UsersSocialNetworksTable::class);
        $select->join($usersSocialNetworksTable->tableGateway->table,
            $this->on($this->column('id'), $usersSocialNetworksTable->column('id_usuario')), [
                'frmUidUsuariosRedesSociales' => 'uid',
                'frmIdRedSocialUsuariosRedesSociales' => 'id_red_social'
            ], JOIN::JOIN_LEFT);
        
        return $this->selectCollection($this->tableGateway, $select, $params);
    }
    
}