<?php
namespace Application\Model\Catalog\UsersData;

use Application\Library\Complements\MyLogger;
use Application\Library\Model\Table;
use Application\Model\Security\Users\UsersTable;
use Laminas\Db\TableGateway\TableGateway;
use Laminas\ServiceManager\ServiceManager;
use Laminas\Db\Sql\Select;
use Laminas\Db\Sql\Join;
use Laminas\Db\Sql\Update;
use Laminas\Db\Sql\Insert;
use Laminas\Db\Sql\Expression;

class UsersDataTable extends Table
{
    /**
     * Constructor
     * @param TableGateway $tableGateway
     * @param ServiceManager $sm
     */
    public function __construct($tableGateway, $sm)
    {
        parent::__construct($tableGateway, $sm);
    }
    
    /**
     * Fetch all data.
     * @param array $params
     * @param \Countable $resultSetPrototype
     * @return \Countable
     */
    public function fetchAll($params = [])
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdEmpleado' => 'id',
            'frmNombreEmpleado' => $this->columnUpper('nombre', true),
            'frmApellidosEmpleado' => $this->columnUpper('apellidos', true),
            'frmCorreoElectronicoEmpleado' => 'correo_electronico',
            'frmTelefonoEmpleado' => 'telefono',
            'frmFechaNacimientoEmpleado' => 'fecha_nacimiento',
            'frmCelularEmpleado' => 'celular',
            'frmIdColoniaEmpleado' => 'id_colonia',
            'frmCalleEmpleado' => $this->columnUpper('calle', true),
            'frmNumeroExteriorEmpleado' => $this->columnUpper('numero_exterior', true),
            'frmNumeroInteriorEmpleado' => $this->columnUpper('numero_interior', true),
            'frmHiddenEmployee' => 'hidden',
            'frmActivoEmpleado' => 'activo',
        ]);
        
        $usersTable = $this->sm->get(UsersTable::class);
        $select->join($usersTable->tableGateway->table,
            $this->on($this->column('id'), $usersTable->column('id_empleado')), [
                'frmIdUsuario' => 'id',
                'frmIdNombreUsuarioUsuario' => 'nombre_usuario'
        ], Join::JOIN_LEFT);
        
//         $neighborhoodsTable = $this->sm->get(NeighborhoodsTable::class);
//         $select->join($neighborhoodsTable->tableGateway->table,
//             $this->on($this->column('id_colonia'), $neighborhoodsTable->column('id')), [
//                 'frmNombreColonia' => 'nombre'
//             ], Join::JOIN_LEFT);
        
//         $zipCodesTable = $this->sm->get(ZipCodesTable::class);
//         $select->join($zipCodesTable->tableGateway->table,
//             $this->on($neighborhoodsTable->column('id_codigo_postal'), $zipCodesTable->column('id')), [
//                 'frmNombreCodigoPostal' => 'nombre'
//             ], Join::JOIN_LEFT);
        
        $select->order($this->asc('nombre'));
        $select->order($this->asc('apellidos'));
        
        return $this->selectCollection($this->tableGateway, $select, $params);
    }
    
    /**
     * Fetch a resource.
     * @param int $id
     * @return \Application\Library\Model\Result
     */
    public function fetch($id)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdEmpleado' => 'id',
            'frmNombreEmpleado' => $this->columnUpper('nombre', true),
            'frmApellidosEmpleado' => $this->columnUpper('apellidos', true),
            'frmCorreoElectronicoEmpleado' => 'correo_electronico',
            'frmSignatureEmployee' => 'signature',
            'frmFrontIneEmployee' => 'front_ine',
            'frmBackIneEmployee' => 'back_ine',
            'frmSectionNameEmployee' => $this->columnToFour('section_name')
        ]);
        
        $select->where->equalTo($this->column('id'), $id);

        return $this->selectEntity($this->tableGateway, $select);
    }
    
    /**
     * Creates resource.
     * @return mixed
     */
    public function create($data)
    {
        try {
            $this->connection->beginTransaction();
            $insert = new Insert($this->tableGateway->table);
            $insert->values([
                'nombre' => $data['frmNombreUsuario'],
                'apellidos' => $data['frmApellidosUsuario'],
                'telefono' => $data['frmTelefonoUsuario'],
                'celular' => $data['frmCelularUsuario'],
                'correo_electronico' => $data['frmCorreoElectronicoUsuario'],
                'fecha_nacimiento' => $data['frmFechaNacimientoUsuario'],
                'id_colonia' => 1,
                'calle' => null,
                'numero_exterior' => 1,
                'numero_interior' => 1,
                'id_usuario_creacion' => 1,
                'fecha_creacion' => date('Y-m-d'),
                'hora_creacion' => date('H:i:s'),
            ]);
            $id = $this->insertWith($insert);

            // Creates user for acces in system
            $securityData = [
                'frmIdUsuarioCatalogo' => $id,
                'frmIdPerfilUsuario' => '1',
                'frmNombreUsuario' => $data['frmCorreoElectronicoUsuario'],
                'frmPasswordUsuario' => $data['frmContraseñaUsuario'],
                'frmUidRedSocialUsuario' => $data['']
            ];

            $idUser = $this->sm->get(UsersTable::class)->createFromWeb($securityData);

            $this->connection->commit();
            return $idUser;
        } catch (\Exception $e) {
            $this->connection->rollback();
            throw new \Exception($e->getMessage());
        }
    }
    
    /**
     * Updates a resource.
     * @param int $id
     * @param array $data
     * @throws \Exception
     * @return mixed
     */
    public function update($id, $data)
    {
        try {
            $this->connection->beginTransaction();
            
            $update = new Update($this->tableGateway->table);
            $update->set([
                'voter_id' => $data['frmVoterIdEmployee'],
                'nombre' => $data['frmNombreEmpleado'],
                'apellidos' => $data['frmApellidosEmpleado'],
                'telefono' => $data['frmTelefonoEmpleado'],
                'celular' => $data['frmCelularEmpleado'],
                'correo_electronico' => $data['frmCorreoElectronicoEmpleado'],
                'fecha_nacimiento' => $data['frmFechaNacimientoEmpleado'],
                'id_colonia' => $data['frmIdColoniaEmpleado'],
                'calle' => $data['frmCalleEmpleado'],
                'numero_exterior' => $data['frmNumeroExteriorEmpleado'],
                'numero_interior' => $data['frmNumeroInteriorEmpleado'],
                'signature' => $data['frmSignatureEmployee'],
                'front_ine' => $data['frmFrontIneEmployee'],
                'back_ine' => $data['frmBackIneEmployee'],
                'id_usuario_modificacion' => $_SESSION['user']['id'],
                'fecha_modificacion' => date('Y-m-d'),
                'hora_modificacion' => date('H:i:s'),
                'section_name' => $data['frmSectionNameEmployee']
            ]);
            $update->where->equalTo($this->column('id'), $id);
            $this->tableGateway->updateWith($update);
            
            $this->connection->commit();
            return $id;
        } catch (\Exception $e) {
            $this->connection->rollback();
            throw new \Exception($e->getMessage());
        }
    }
    
    /**
     * Select by ID.
     * @param int $id
     * @return \Laminas\Db\ResultSet\ResultSetInterface
     */
    public function selectById($id)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdEmpleado' => 'id',
            'frmNombreEmpleado' => 'nombre',
            'frmApellidosEmpleado' => 'apellidos',
            'frmCorreoElectronicoEmpleado' => 'correo_electronico'
        ]);
        
        $select->where->equalTo($this->column('id'), $id);
        
        return $this->tableGateway->selectWith($select)->current();
    }
    
    /**
     * Select by voter ID.
     * @param int $id
     * @return \Laminas\Db\ResultSet\ResultSetInterface
     */
    public function selectByVoterId($voterId, $idOmit = null)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdEmpleado' => 'id',
            'frmNombreEmpleado' => 'nombre',
            'frmApellidosEmpleado' => 'apellidos',
            'frmCorreoElectronicoEmpleado' => 'correo_electronico'
        ]);
        
        $select->where->equalTo($this->column('voter_id', true), $voterId);
        if($idOmit) {
            $select->where->notEqualTo($this->column('id'), $idOmit);
        }
        
        return $this->tableGateway->selectWith($select)->current();
    }

    /**
     * Select by name.
     * @param string $name
     * @param string $lastName
     * @param string $maternalSurname
     * @param string $idOmit
     * @return mixed
     */
    public function selectByName($name, $lastName, $idOmit = null)
    {
        $select = new Select($this->tableGateway->table);
        $select->columns([
            'frmIdEmpleado' => 'id',
            'frmNombreEmpleado' => 'nombre',
            'frmApellidosEmpleado' => 'apellidos',
        ]);
        $select->where->equalTo($this->column('nombre', true), $this->unaccentLowercaseValue($name));
        $select->where->equalTo($this->column('apellidos', true), $this->unaccentLowercaseValue($lastName));
        if($idOmit) {
            $select->where->notEqualTo($this->column('id'), $idOmit);
        }
        return $this->tableGateway->selectWith($select)->current();
    }
}